//
// $Id: fmcmos.cc,v 1.3 2001/07/24 12:38:01 nishi Exp $
//
// Copyright (C) 2001 Shouhei Nishi.
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
// 3. The name of author may not be used to endorse or promote products
//    derived from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
// OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
// SUCH DAMAGE.
//





#include "bochs.h"
#if BX_EMULATION_TOWNS
#define LOG_THIS bx_cmos.

bx_cmos_c bx_cmos;

#if BX_USE_CMOS_SMF
#define this (&bx_cmos)
#endif


// check that BX_NUM_CMOS_REGS is 64 or 128
#if (BX_NUM_CMOS_REGS == (8*1024))
#else
#error "Invalid BX_NUM_CMOS_REGS value in config.h"
#endif


bx_cmos_c::bx_cmos_c(void)
{
	setprefix("[CMOS]");
	settype(CMOSLOG);
	BX_DEBUG(("Init.\n"));
}

bx_cmos_c::~bx_cmos_c(void)
{
  // nothing for now
  BX_DEBUG(("Exit.\n"));
}


  void
bx_cmos_c::init(bx_devices_c *d)
{
  unsigned i;

  // CMOS RAM & RTC

  BX_CMOS_THIS devices = d;

  for(i=0x3000;i<0x4000;i+=2) {
    BX_CMOS_THIS devices->register_io_read_handler(this,
						   read_handler, i,
						   "CMOS RAM");
    BX_CMOS_THIS devices->register_io_write_handler(this,
						    write_handler, i,
						    "CMOS RAM");
  }
  BX_CMOS_THIS devices->register_io_read_handler(this,
						 read_handler, 0x70,
						 "CMOS RTC");
  BX_CMOS_THIS devices->register_io_write_handler(this,
						  write_handler, 0x70,
						  "CMOS RTC");
  BX_CMOS_THIS devices->register_io_write_handler(this,
						  write_handler, 0x80,
						  "CMOS RTC");

  BX_CMOS_THIS s.one_second_timer_index =
    bx_pc_system.register_timer(this, one_second_timer_handler,
      1000000/2, 1,0); // continuous, not-active

#if BX_USE_SPECIFIED_TIME0 == 0
  // ??? this will not be correct for using an image file.
  // perhaps take values in CMOS and work backwards to find
  // s.timeval from values read in.
  BX_CMOS_THIS s.timeval = time(NULL);
#else
  BX_CMOS_THIS s.timeval = BX_USE_SPECIFIED_TIME0;
#endif

  if (bx_options.cmos.time0 == 1)
          BX_CMOS_THIS s.timeval = time(NULL);
  else if (bx_options.cmos.time0 != 0)
          BX_CMOS_THIS s.timeval = bx_options.cmos.time0;

  BX_INFO(("Setting initial clock to: %s",
    ctime(&(BX_CMOS_THIS s.timeval)) ));

  update_clock();

  // load CMOS from image file if requested.
  if (bx_options.cmos.cmosImage) {
    // CMOS image file requested
    int fd, ret;
    struct stat stat_buf;

    fd = open(bx_options.cmos.path,
#if HAVE_MMAP && HAVE_MAP_FAILED
	      O_RDWR
#else
	      O_RDONLY
#endif
#ifdef O_BINARY
            | O_BINARY
#endif
             );
    if (fd < 0) {
      perror("trying to open cmos image file.\n");
      BX_INFO(("trying to creat new cmos image file '%s'\n",
        bx_options.cmos.path));
      fd = open(bx_options.cmos.path,
		O_RDWR | O_CREAT | O_TRUNC
#ifdef O_BINARY
		| O_BINARY
#endif
//		,
//		S_IRUSR | S_IWUSR
#ifdef S_IRGRP
		| S_IRGRP
#endif
#ifdef S_IROTH
		| S_IROTH
#endif
		);
      if(fd < 0) {
	perror("trying to creat new cmos image file.\n");
	BX_PANIC(("trying to creat new cmos image file '%s'\n",
		  bx_options.cmos.path));
        }
      {
	int i;
	Bit8u d=0;
	for(i=0;i<BX_NUM_CMOS_REGS;i++) {
	  if(::write(fd, &d, 1) != 1) {
	    BX_PANIC(("trying to creat new cmos image file '%s'\n",
		      bx_options.cmos.path));
	  }
	}
	lseek(fd, 0, SEEK_SET);
      }
    }

    ret = fstat(fd, &stat_buf);
    if (ret) {
      BX_PANIC(("CMOS: could not fstat() image file.\n"));
      }
    if (stat_buf.st_size != BX_NUM_CMOS_REGS) {
      BX_PANIC(("CMOS: image file not same size as BX_NUM_CMOS_REGS.\n"));
      }

#if HAVE_MMAP && HAVE_MAP_FAILED
    if((BX_CMOS_THIS s.reg = (Bit8u*)mmap(NULL,
					  BX_NUM_CMOS_REGS,
					  PROT_READ | PROT_WRITE,
					  MAP_SHARED,
					  fd,
					  0))==MAP_FAILED) {
      BX_PANIC(("CMOS: error memory mapping cmos file.\n"));
      }
    BX_INFO(("CMOS: successfuly mapped from image file '%s'.\n",
      bx_options.cmos.path));
#else
    if((BX_CMOS_THIS s.reg = new Bit8u[BX_NUM_CMOS_REGS])==NULL) {
      BX_PANIC(("CMOS: error allocating cmos data.\n"));
    }
    if(read(fd, BX_CMOS_THIS s.reg, BX_NUM_CMOS_REGS) !=
       BX_NUM_CMOS_REGS) {
      BX_PANIC(("CMOS: error read cmos file.\n"));
    }
    BX_INFO(("CMOS: successfuly read from image file '%s'.\n",
	      bx_options.cmos.path));
    close(fd);
#endif
    } else {
      if((BX_CMOS_THIS s.reg = new Bit8u[BX_NUM_CMOS_REGS])==NULL) {
	BX_PANIC(("CMOS: error allocating cmos data.\n"));
      }
    }
}

  void
bx_cmos_c::save(void)
{
#if !(HAVE_MMAP && HAVE_MAP_FAILED)
  // save CMOS from image file if requested.
  if (bx_options.cmos.cmosImage) {
    // CMOS image file requested
    int fd, ret;
    struct stat stat_buf;

    fd = open(bx_options.cmos.path, O_RDWR
#ifdef O_BINARY
            | O_BINARY
#endif
             );
    if (fd < 0) {
      perror("trying to open cmos image file.\n");
      BX_INFO(("trying to open cmos image file '%s'\n",
        bx_options.cmos.path));
      return;
      }

    if(write(fd, BX_CMOS_THIS s.reg, BX_NUM_CMOS_REGS) !=
       BX_NUM_CMOS_REGS) {
      BX_INFO(("CMOS: error write cmos file.\n"));
      return;
    }
    BX_INFO(("CMOS: successfuly write to image file '%s'.\n",
	      bx_options.cmos.path));
    close(fd);
    }
#endif
}

  void
bx_cmos_c::reset(void)
{
  // One second timer for updating clock & alarm functions
  bx_pc_system.activate_timer(BX_CMOS_THIS s.one_second_timer_index,
                              0, 1);
}


  // static IO port read callback handler
  // redirects to non-static class handler to avoid virtual functions

  Bit32u
bx_cmos_c::read_handler(void *this_ptr, Bit32u address, unsigned io_len)
{
#if !BX_USE_CMOS_SMF
  bx_cmos_c *class_ptr = (bx_cmos_c *) this_ptr;

  return( class_ptr->read(address, io_len) );
}

  Bit32u
bx_cmos_c::read(Bit32u address, unsigned io_len)
{
#else
  UNUSED(this_ptr);
#endif
  Bit32u ret;
#define RETURN(x) do { ret = (x); goto read_return; } while (0)

  if (io_len > 1)
    BX_PANIC(("cmos: io read from address %08x len=%u\n",
	      (unsigned) address, (unsigned) io_len));

  if(address >= 0x3000 && address < 0x4000 && (address & 1) == 0) {
    RETURN(BX_CMOS_THIS s.reg[(address-0x3000)/2]);
  }
  switch (address) {
    case 0x0070:
      RETURN(BX_CMOS_THIS s.rtc_data[BX_CMOS_THIS s.rtc_addr] | 0x80 |
	     (BX_CMOS_THIS s.READY ? 0x80 : 0x00));

    default:
      BX_PANIC(("unsupported cmos read, address=0x%x!\n",
        (unsigned) address));
      RETURN(0);
      break;
    }
  read_return:
  if (bx_dbg.cmos)
    BX_INFO(("CMOS read from address: 0x%x = 0x%x\n",
      (unsigned) address, (unsigned) ret));
  return ret;
}
#undef RETURN


  // static IO port write callback handler
  // redirects to non-static class handler to avoid virtual functions

  void
bx_cmos_c::write_handler(void *this_ptr, Bit32u address, Bit32u value, unsigned io_len)
{
#if !BX_USE_CMOS_SMF
  bx_cmos_c *class_ptr = (bx_cmos_c *) this_ptr;

  class_ptr->write(address, value, io_len);
}

  void
bx_cmos_c::write(Bit32u address, Bit32u value, unsigned io_len)
{
#else
  UNUSED(this_ptr);
#endif  // !BX_USE_CMOS_SMF

  if (io_len > 1)
    BX_PANIC(("cmos: io write to address %08x len=%u\n",
             (unsigned) address, (unsigned) io_len));

  if (bx_dbg.cmos)
    BX_INFO(("CMOS write to address: 0x%x = 0x%x\n",
      (unsigned) address, (unsigned) value));

  if(address >= 0x3000 && address < 0x4000 && (address & 1) == 0) {
    BX_CMOS_THIS s.reg[(address-0x3000)/2]=value;
    return;
  }


  switch (address) {
  case 0x0070:
    BX_CMOS_THIS s.rtc_datareg = value;
    return;

  case 0x0080:
    if((value&0x80)==0) return;
    if((value&0x01)!=0) {
      BX_CMOS_THIS s.rtc_addr = BX_CMOS_THIS s.rtc_datareg & 0xf;
      return;
    }
    return;

  default:
    BX_PANIC(("unsupported cmos write, address=0x%.4x!\n",
	      (unsigned) address));
    return;
    break;
  }
}


  void
bx_cmos_c::one_second_timer_handler(void *this_ptr)
{
  bx_cmos_c *class_ptr = (bx_cmos_c *) this_ptr;

  if(class_ptr->s.READY) {
    class_ptr->s.READY=1;
    return;
  }
  class_ptr->s.READY=0;

  // update internal time/date buffer
  class_ptr->s.timeval++;

#if BX_GETTIMEOFDAY
  if (bx_dbg.cmos) {
    struct timeval tv;

    gettimeofday(&tv,NULL);

    BX_INFO(("cmos: gettimeofday=(%ld, %ld)\n",
	     tv.tv_sec,
	     tv.tv_usec));
  }
#endif

  class_ptr->update_clock();
}


  void
bx_cmos_c::update_clock()
{
  struct tm *time_calendar;

  time_calendar = localtime(& BX_CMOS_THIS s.timeval);

  memset(BX_CMOS_THIS s.rtc_data,0,16);

  // update seconds

  BX_CMOS_THIS s.rtc_data[0x0] =   time_calendar->tm_sec      % 10;
  BX_CMOS_THIS s.rtc_data[0x1] =   time_calendar->tm_sec      / 10;

  // update minutes
  BX_CMOS_THIS s.rtc_data[0x2] =   time_calendar->tm_min      % 10;
  BX_CMOS_THIS s.rtc_data[0x3] =   time_calendar->tm_min      / 10;

  // update hours
  BX_CMOS_THIS s.rtc_data[0x4] =   time_calendar->tm_hour     % 10;
  BX_CMOS_THIS s.rtc_data[0x5] = ( time_calendar->tm_hour     / 10) | 0x08;

  // update day of the week
  BX_CMOS_THIS s.rtc_data[0x6] =   time_calendar->tm_wday;

  // update day of the month
  BX_CMOS_THIS s.rtc_data[0x7] =   time_calendar->tm_mday     % 10;
  BX_CMOS_THIS s.rtc_data[0x8] = ( time_calendar->tm_mday     / 10) | 0x00;

  // update month
  BX_CMOS_THIS s.rtc_data[0x9] =  (time_calendar->tm_mon + 1) % 10;
  BX_CMOS_THIS s.rtc_data[0xa] = ((time_calendar->tm_mon + 1) / 10) | 0x00;

  // update year
  BX_CMOS_THIS s.rtc_data[0xb] =   time_calendar->tm_year     % 10;
  BX_CMOS_THIS s.rtc_data[0xc] = ( time_calendar->tm_year     / 10) % 10;
}
#endif
